/**
 * \file
 *
 * \brief LORAWAN Demo Application
 *
 * Copyright (c) 2016 Atmel Corporation. All rights reserved.
 *
 * \asf_license_start
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 *    Atmel microcontroller product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * \asf_license_stop
 *
 */
/****************************** INCLUDES **************************************/
#include "asf.h"
#include "lorawan.h"
#include "system_task_manager.h"
#include "enddevice_cert.h"
#include "conf_app.h"
#include "sio2host.h"
#include "sw_timer.h"
/************************** GLOBAL VARIABLES ***********************************/

static uint8_t testMode = OFF;
static uint8_t sendData[100];
static uint16_t sendDataLen;
static uint16_t downlinkCtr = 0;
static bool bTxCnf = false;
static uint8_t mrgn = 255;
static uint8_t gwnb = 0;
static uint8_t uplinkTestNoResp = 0;
bool bLinkCheckEnabled = false;
static bool pktRxd = false;
 
uint8_t certAppTimerId;
LorawanSendReq_t lorawanSendReq;

/*ABP Join Parameters */
uint32_t devAddr = DEVICE_ADDRESS;
uint8_t nwksKey[16] = NETWORK_SESSION_KEY;
uint8_t appsKey[16] = APPLICATION_SESSION_KEY;

/* OTAA join parameters */
uint8_t devEui[8] = DEVICE_EUI;
uint8_t appEui[8] = APPLICATION_EUI;
uint8_t appKey[16] = APPLICATION_KEY;

/************************** EXTERN VARIABLES ***********************************/


/************************** FUNCTION PROTOTYPES ********************************/
SYSTEM_TaskStatus_t APP_TaskHandler(void);

static void EnterTestMode(void);

static void certAppTimerCb(uint8_t param);

static void handleCertRxData(uint8_t* data,uint8_t dataLen);
/***************************** FUNCTIONS ***************************************/

/*********************************************************************//**
\brief	Initialization the Demo application 
*************************************************************************/
void cert_app_init(void)
{
	StackRetStatus_t status;
	/* Initialize the Serial Interface */
	sio2host_init();
	
	status = SwTimerCreate(&certAppTimerId);
	
	if(status!=LORAWAN_SUCCESS)
	{
	//	SYS_ASSERT_ERROR(0);
	while(1);//todo remove
	}
	
	SwTimerSetCallback (certAppTimerId, certAppTimerCb, 0);
	
	/* Initialize the LORAWAN Stack */
	LORAWAN_Init(cert_appdata_callback, cert_joindata_callback,ISM_EU868);
	
	printf("\n\n\r*******************************************************\n\r");
	printf("\n\rMicrochip LoRaWAN Stack %s\r\n",STACK_VER); 
	printf("\r\nInit - Successful\r\n");
	
	status = set_mac_parameters();
	
	if (LORAWAN_SUCCESS != status)
	{
		printf("\nMAC parameters initialization failed\n\r");
	}
	/* Initialize the join parameters */
	status = set_join_parameters(LORAWAN_ABP);
	if (LORAWAN_SUCCESS != status)
	{
		printf("\nJoin parameters initialization failed\n\r");
	}
	
	//TODO : Post to apptask - set the join parameters and call LORAWAN_Join when otaa test is triggered 
	//below is temp code to verify tc
	status = set_join_parameters(LORAWAN_OTAA);
	if (LORAWAN_SUCCESS != status)
	{
		printf("\nOTAA Join parameters initialization failed\n\r");
	}
	/* Send Join request */
	status = LORAWAN_Join(APP_ACTIVATION_TYPE);
	if (status == LORAWAN_SUCCESS)
	{
		printf("\nJoin Request Sent\n\r");
	}
	else
	{
		print_stack_status(status);
	}

}


/*********************************************************************//**
\brief Callback function for the ending of Bidirectional communication of 
       Application data 
 *************************************************************************/
void cert_appdata_callback(void *appHandle, appCbParams_t *appdata)
{
	
	if (LORAWAN_EVT_RX_DATA_AVAILABLE == appdata->evt)
	{
		uint8_t *pData = appdata->param.rxData.pData;
		uint8_t dataLength = appdata->param.rxData.dataLength;
		OpStatus_t status = appdata->param.rxData.status;
		uint32_t devAddress = appdata->param.rxData.devAddr;
		
		switch(status)
		{
			case MAC_OK:
			{
				 pktRxd = true;
				 //Successful transmission
				 if((dataLength > 0U) && (NULL != pData))
				 {
					
					 printf("*** Received DL Data ***\n\r");
					 printf("\nFrame Received at port %d\n\r",pData[0]);
					 printf("\nFrame Length - %d\n\r",dataLength);
					 printf("\nAddress - 0x%lx\n\r", devAddress);
					 printf ("\nPayload: ");
					 for (uint8_t i =0; i<dataLength - 1; i++)
					 {
						 printf("%x",pData[i+1]);
					 }
					 printf("\r\n*************************\r\n");
					
					 if(pData[0] == TEST_PORT_NB)
					 {
					   handleCertRxData(pData+1,dataLength-1);
					 }
				 }
				 else
				 {
					 uplinkTestNoResp++;
					 printf("Received ACK for Confirmed data\r\n");
				 }
				 uplinkTestNoResp = 0;
				 
			}	
			break;

									
			default:
				printf("\r\nMAC NOK!");
			break;
		}
	}
	else if(LORAWAN_EVT_TRANSACTION_COMPLETE == appdata->evt)
	{
		switch(appdata->param.transCmpl.status)
		{
			case MAC_OK:
			{
				if(pktRxd == false)
				{ 
					printf("Transmission Success\r\n");
					// No answer received from the server side
					// Increment no response counter
					uplinkTestNoResp++;
					//prepare the next data
					sendData[0] = (uint8_t) (downlinkCtr >> 8);
					sendData[1] = (uint8_t) (downlinkCtr);
					sendDataLen = 2;
				}
			}
			break;
						
			default:
				printf("\r\nMAC NOK!");
			break;
		}
		printf("\n\r*************************************************\n\r");
	}
		
}

/*********************************************************************//*
\brief Callback function for the ending of Activation procedure 
 ************************************************************************/
void cert_joindata_callback(bool status)
{
	/* This is called every time the join process is finished */
	if(true == status)
	{
		printf("\nJoining Successful\n\r");	
		
		//EnterTestMode();
		/*Start the Timer to send data periodically*/
		SwTimerSetTimeout (certAppTimerId, MS_TO_TICKS(CERT_APP_TIMEOUT));
		SwTimerStart (certAppTimerId);
	}
	else
	{
		printf("\nJoining Denied\n\r");
		
		printf("\nReset the Device\n\r");
		
		while(1);
	}
	printf("\n\r*******************************************************\n\r");	
}

/*********************************************************************//*
 \brief  Application Task Handler 
 \return processing status
 ************************************************************************/
SYSTEM_TaskStatus_t APP_TaskHandler(void)
{	
	return SYSTEM_TASK_SUCCESS;
}

/*********************************************************************//*
 \brief      Set MAC  parameters function
 \return	 LORAWAN_SUCCESS, if successfully set the join parameters
             LORAWAN_INVALID_PARAMETER, otherwise
 ************************************************************************/
StackRetStatus_t set_mac_parameters(void)
{
	StackRetStatus_t status = LORAWAN_SUCCESS;
	ChannelParameters_t chParam;
	bool adrValue = true ; 
	bool dcEnable = false;
/*
	for(uint8_t i=0 ; i<3 && (status == LORAWAN_SUCCESS);i++)
	{		
		chParam.channelId = i;
		chParam.channelAttr.dutyCycle = 0;
		status = LORAWAN_SetAttr(CH_PARAM_DUTYCYCLE,&chParam);
	}	*/
	status = LORAWAN_SetAttr(DUTY_CYCLE_ENABLE,&dcEnable);
	
	if(status == LORAWAN_SUCCESS)
	{
		status = LORAWAN_SetAttr(ADR,&adrValue);
	}
	
	return status;
	
}


/*********************************************************************//*
 \brief      Set join parameters function
 \param[in]  activation type - notifies the activation type (OTAA/ABP)
 \return	 LORAWAN_SUCCESS, if successfully set the join parameters
             LORAWAN_INVALID_PARAMETER, otherwise
 ************************************************************************/
StackRetStatus_t set_join_parameters(ActivationType_t activation_type)
{
	StackRetStatus_t status;
	
	printf("\n********************Join Parameters********************\n\r");
	
	if(ACTIVATION_BY_PERSONALIZATION == activation_type)
	{
		status = LORAWAN_SetAttr (DEV_ADDR, &devAddr);
		if (LORAWAN_SUCCESS == status)
		{
			status = LORAWAN_SetAttr (APPS_KEY, appsKey);
		}
		
		if (LORAWAN_SUCCESS == status)
		{
			printf("\nApplication Session Key - ");
			print_array((uint8_t *)&appsKey, sizeof(appsKey));
			status = LORAWAN_SetAttr (NWKS_KEY, nwksKey);
		}
		
		if (LORAWAN_SUCCESS == status)
		{
			printf("\nNetwork Session Key - ");
			print_array((uint8_t *)&nwksKey, sizeof(nwksKey));
		}
					
	}
	else
	{
		status = LORAWAN_SetAttr (DEV_EUI, devEui);
		if (LORAWAN_SUCCESS == status)
		{
			printf("\nDevice EUI - ");
			print_array((uint8_t *)&devEui, sizeof(devEui));
			status = LORAWAN_SetAttr (APP_EUI, appEui);
		}
		
		if (LORAWAN_SUCCESS == status)
		{
			printf("\nApplication EUI - ");
			print_array((uint8_t *)&appEui, sizeof(appEui));
			status = LORAWAN_SetAttr (APP_KEY, appKey);
		}
		
		if (LORAWAN_SUCCESS == status)
		{
			printf("\nApplication Key - ");
			print_array((uint8_t *)&appKey, sizeof(appKey));
		}	
	}
	return status;
}



/*********************************************************************//*
 \brief      Function to Print array of characters
 \param[in]  *array  - Pointer of the array to be printed
 \param[in]   length - Length of the array
 ************************************************************************/
void print_array (uint8_t *array, uint8_t length)
{
	printf("0x");
	for (uint8_t i =0; i < length; i++)
	{
		printf("%02x", *array);
		array++;
	}
	printf("\n\r");
}



/*********************************************************************//*
 \brief      Function to Print stack return status
 \param[in]  status - Status from the stack
 ************************************************************************/
void print_stack_status(StackRetStatus_t status)
{
	switch(status)
	{
		case LORAWAN_SUCCESS:
		     printf("\nLorawan Success\n\r");
		break;
		
		case LORAWAN_BUSY:
		     printf("\nLorawan state invalid - Stack Busy\n\r");
		break;
			 
		case LORAWAN_NWK_NOT_JOINED:
		    printf("\nDevice not joined to network yet\n\r");
		break;
		
		case LORAWAN_INVALID_PARAMETER:
			printf("\nInvalid parameter\n\r");
		break;
		
		case LORAWAN_KEYS_NOT_INITIALIZED:
			printf("\nKeys not initialized\n\r");
		break;
		
		case LORAWAN_SILENT_IMMEDIATELY_ACTIVE:
			printf("\nSilent immediately active\n\r");
		break;
		
		case LORAWAN_FCNTR_ERROR_REJOIN_NEEDED:
			printf("\nFramecounter Error - Rejoin needed\n\r");
		break;
		
		case LORAWAN_INVALID_BUFFER_LENGTH:
			printf("\nInvalid buffer length\n\r");
		break;
		
		case LORAWAN_MAC_PAUSED:
			printf("\nMAC Paused\n\r");
		break;
		
		case LORAWAN_NO_CHANNELS_FOUND:
			printf("\nNo free channels found\n\r");
		break;
		
		case LORAWAN_INVALID_REQUEST:
			printf("\nRequest invalid\n\r");
		break;
	    
		default:
		   printf("\nRequest Failed\n\r");
		break;
	}
}

static void certAppTimerCb(uint8_t param)
{
	StackRetStatus_t status;
	SwTimerSetTimeout (certAppTimerId, MS_TO_TICKS(CERT_APP_TIMEOUT));
	SwTimerStart (certAppTimerId);	
	if(testMode == OFF)
	{		
		lorawanSendReq.buffer = sendData;
		lorawanSendReq.bufferLength = sendDataLen;
		lorawanSendReq.confirmed = APP_TRANSMISSION_TYPE;
		lorawanSendReq.port = APP_FPORT;
		status = LORAWAN_Send(&lorawanSendReq);	
	}
	else
	{
		if (uplinkTestNoResp < UPLINK_NO_RESPONSE_NB) {
			
			lorawanSendReq.buffer = sendData;
			lorawanSendReq.bufferLength = sendDataLen;
			lorawanSendReq.confirmed = bTxCnf;
			lorawanSendReq.port = TEST_PORT_NB;
			status = LORAWAN_Send(&lorawanSendReq);	
		}
		else
		{
			 //Deactivated test mode after
			 testMode = OFF;
			 return;
		}
	}

			
	if (LORAWAN_SUCCESS == status)
	{
		pktRxd = false;
		printf("\nFrame Sent:");
		print_array(lorawanSendReq.buffer,lorawanSendReq.bufferLength);
	}
	else
	{
		print_stack_status(status);
	}	

}

static void EnterTestMode(void) 
{
	testMode = ON;
	downlinkCtr = 0;
	uplinkTestNoResp = 0;

	//prepare the first data
    sendData[0] = (uint8_t) (downlinkCtr >> 8);
    sendData[1] = (uint8_t) (downlinkCtr);
    sendDataLen = 2;
	printf("\r\nTest Mode Activated\r\n");
}

static void handleCertRxData(uint8_t* data,uint8_t dataLen)
{
 StackRetStatus_t status;
 if (bLinkCheckEnabled) 
 {
/*
	 bLinkCheckEnabled = false;

	 sio2ncp_tx("mac set linkchk 0\r\n",19);
	 sio2host_tx("mac set linkchk 0\r\n",19);
	 uartRxByte = EUSART1_Read();
	 ClearReceptionBuffer();

	 sio2ncp_tx("mac get mrgn\r\n",14);
	 sio2host_tx("mac get mrgn\r\n",14);
	 mrgn = ReadUartConvertToUint8('\r');
	 ClearReceptionBuffer();

	 sio2ncp_tx("mac get gwnb\r\n",14);
	 sio2host_tx("mac get gwnb\r\n",14);
	 gwnb = ReadUartConvertToUint8('\r');
	 ClearReceptionBuffer();

	 sendData[0] = LINK_CHECK_MODE;
	 sendData[1] = LINK_CHECK_ANS_CID;
	 sendData[2] = mrgn;
	 sendData[3] = gwnb;
	 sendDataLen = 4;*/
	 }
	 else
	 {
		 if (dataLen > 0) 
		 {
			downlinkCtr++;
			uplinkTestNoResp = 0;

			//prepare the next data
			sendData[0] = (uint8_t) (downlinkCtr >> 8);
			sendData[1] = (uint8_t) (downlinkCtr);
			sendDataLen = 2;		
				
			switch (data[0]) {
				case DEACTIVATE_MODE:
				
				if (dataLen == 1) {
					//Deactivated test mode
					testMode = OFF;
				}
				break;

				case ACTIVATE_MODE:
				{
					bool bPayloadValid = false;
					if (dataLen == 4) {
						for (uint8_t i = 0; i < 4; i++) {
							if (data[i] == 1) {
								bPayloadValid = true;
								} else {
								bPayloadValid = false;
								break;
							}
						}
					}
					
					if (bPayloadValid) {
						//Activated test mode
						EnterTestMode();
					}
				}
				break;

				case CNF_MODE:
				if (dataLen == 1) {
					//Confirmed frames
					bTxCnf = true;
				}
				break;

				case UNCNF_MODE:
				if (dataLen == 1) {
					//Unconfirmed frames
					bTxCnf = false;
				}
				break;

				case CRYPTO_MODE:
				if (dataLen <= 33) {
					//Cryptography tests
					sendData[0] = 0x04;

					for (uint8_t i = 1; i < dataLen; i++) {
						//TODO: take care of this
						sendData[i] = (((uint16_t) data[i]) + 1) % 256;
					}
					sendDataLen = dataLen;
				}

				break;
				
				case OTAA_TRIGGER_MODE:
				{
					//TODO : Post to apptask - set the join parameters and call LORAWAN_Join ,
					//below is temp code to verify tc,will be changed to task later
					/* Send Join request */
					status = LORAWAN_Join(LORAWAN_OTAA);
					if (status == LORAWAN_SUCCESS)
					{
						testMode = OFF;
						SwTimerStop(certAppTimerId);
						printf("\nOTAA Join Request Sent\n\r");
					}
					else
					{
						print_stack_status(status);
					}
					
				}				
				break;
				
				case LINK_CHECK_MODE:
				if (dataLen == 1) {
					//Generate LinkCheckReq
					//TODO					
					bLinkCheckEnabled = true;
				}
				break;

				default:
				break;
			}
		 }	
	}
}
/* eof demo_app.c */
